/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { getClosureSafeProperty } from '../../util/property';
/**
 * Construct an injectable definition which defines how a token will be constructed by the DI
 * system, and in which injectors (if any) it will be available.
 *
 * This should be assigned to a static `ɵprov` field on a type, which will then be an
 * `InjectableType`.
 *
 * Options:
 * * `providedIn` determines which injectors will include the injectable, by either associating it
 *   with an `@NgModule` or other `InjectorType`, or by specifying that this injectable should be
 *   provided in the `'root'` injector, which will be the application-level injector in most apps.
 * * `factory` gives the zero argument function which will create an instance of the injectable.
 *   The factory can call `inject` to access the `Injector` and request injection of dependencies.
 *
 * @codeGenApi
 * @publicApi This instruction has been emitted by ViewEngine for some time and is deployed to npm.
 */
export function ɵɵdefineInjectable(opts) {
    return {
        token: opts.token,
        providedIn: opts.providedIn || null,
        factory: opts.factory,
        value: undefined,
    };
}
/**
 * @deprecated in v8, delete after v10. This API should be used only by generated code, and that
 * code should now use ɵɵdefineInjectable instead.
 * @publicApi
 */
export const defineInjectable = ɵɵdefineInjectable;
/**
 * Construct an `InjectorDef` which configures an injector.
 *
 * This should be assigned to a static injector def (`ɵinj`) field on a type, which will then be an
 * `InjectorType`.
 *
 * Options:
 *
 * * `providers`: an optional array of providers to add to the injector. Each provider must
 *   either have a factory or point to a type which has a `ɵprov` static property (the
 *   type must be an `InjectableType`).
 * * `imports`: an optional array of imports of other `InjectorType`s or `InjectorTypeWithModule`s
 *   whose providers will also be added to the injector. Locally provided types will override
 *   providers from imports.
 *
 * @codeGenApi
 */
export function ɵɵdefineInjector(options) {
    return { providers: options.providers || [], imports: options.imports || [] };
}
/**
 * Read the injectable def (`ɵprov`) for `type` in a way which is immune to accidentally reading
 * inherited value.
 *
 * @param type A type which may have its own (non-inherited) `ɵprov`.
 */
export function getInjectableDef(type) {
    return getOwnDefinition(type, NG_PROV_DEF) || getOwnDefinition(type, NG_INJECTABLE_DEF);
}
export function isInjectable(type) {
    return getInjectableDef(type) !== null;
}
/**
 * Return definition only if it is defined directly on `type` and is not inherited from a base
 * class of `type`.
 */
function getOwnDefinition(type, field) {
    return type.hasOwnProperty(field) ? type[field] : null;
}
/**
 * Read the injectable def (`ɵprov`) for `type` or read the `ɵprov` from one of its ancestors.
 *
 * @param type A type which may have `ɵprov`, via inheritance.
 *
 * @deprecated Will be removed in a future version of Angular, where an error will occur in the
 *     scenario if we find the `ɵprov` on an ancestor only.
 */
export function getInheritedInjectableDef(type) {
    const def = type && (type[NG_PROV_DEF] || type[NG_INJECTABLE_DEF]);
    if (def) {
        const typeName = getTypeName(type);
        ngDevMode &&
            console.warn(`DEPRECATED: DI is instantiating a token "${typeName}" that inherits its @Injectable decorator but does not provide one itself.\n` +
                `This will become an error in a future version of Angular. Please add @Injectable() to the "${typeName}" class.`);
        return def;
    }
    else {
        return null;
    }
}
/** Gets the name of a type, accounting for some cross-browser differences. */
function getTypeName(type) {
    // `Function.prototype.name` behaves differently between IE and other browsers. In most browsers
    // it'll always return the name of the function itself, no matter how many other functions it
    // inherits from. On IE the function doesn't have its own `name` property, but it takes it from
    // the lowest level in the prototype chain. E.g. if we have `class Foo extends Parent` most
    // browsers will evaluate `Foo.name` to `Foo` while IE will return `Parent`. We work around
    // the issue by converting the function to a string and parsing its name out that way via a regex.
    if (type.hasOwnProperty('name')) {
        return type.name;
    }
    const match = ('' + type).match(/^function\s*([^\s(]+)/);
    return match === null ? '' : match[1];
}
/**
 * Read the injector def type in a way which is immune to accidentally reading inherited value.
 *
 * @param type type which may have an injector def (`ɵinj`)
 */
export function getInjectorDef(type) {
    return type && (type.hasOwnProperty(NG_INJ_DEF) || type.hasOwnProperty(NG_INJECTOR_DEF)) ?
        type[NG_INJ_DEF] :
        null;
}
export const NG_PROV_DEF = getClosureSafeProperty({ ɵprov: getClosureSafeProperty });
export const NG_INJ_DEF = getClosureSafeProperty({ ɵinj: getClosureSafeProperty });
// We need to keep these around so we can read off old defs if new defs are unavailable
export const NG_INJECTABLE_DEF = getClosureSafeProperty({ ngInjectableDef: getClosureSafeProperty });
export const NG_INJECTOR_DEF = getClosureSafeProperty({ ngInjectorDef: getClosureSafeProperty });
//# sourceMappingURL=data:application/json;base64,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