/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { concat, defer, from, of, pipe } from 'rxjs';
import { concatMap, first, map, mergeMap, tap } from 'rxjs/operators';
import { ActivationStart, ChildActivationStart } from '../events';
import { redirectingNavigationError } from '../navigation_canceling_error';
import { isUrlTree } from '../url_tree';
import { wrapIntoObservable } from '../utils/collection';
import { getClosestRouteInjector } from '../utils/config';
import { getCanActivateChild, getTokenOrFunctionIdentity } from '../utils/preactivation';
import { isBoolean, isCanActivate, isCanActivateChild, isCanDeactivate, isCanLoad, isCanMatch } from '../utils/type_guards';
import { prioritizedGuardValue } from './prioritized_guard_value';
export function checkGuards(injector, forwardEvent) {
    return mergeMap(t => {
        const { targetSnapshot, currentSnapshot, guards: { canActivateChecks, canDeactivateChecks } } = t;
        if (canDeactivateChecks.length === 0 && canActivateChecks.length === 0) {
            return of({ ...t, guardsResult: true });
        }
        return runCanDeactivateChecks(canDeactivateChecks, targetSnapshot, currentSnapshot, injector)
            .pipe(mergeMap(canDeactivate => {
            return canDeactivate && isBoolean(canDeactivate) ?
                runCanActivateChecks(targetSnapshot, canActivateChecks, injector, forwardEvent) :
                of(canDeactivate);
        }), map(guardsResult => ({ ...t, guardsResult })));
    });
}
function runCanDeactivateChecks(checks, futureRSS, currRSS, injector) {
    return from(checks).pipe(mergeMap(check => runCanDeactivate(check.component, check.route, currRSS, futureRSS, injector)), first(result => {
        return result !== true;
    }, true));
}
function runCanActivateChecks(futureSnapshot, checks, injector, forwardEvent) {
    return from(checks).pipe(concatMap((check) => {
        return concat(fireChildActivationStart(check.route.parent, forwardEvent), fireActivationStart(check.route, forwardEvent), runCanActivateChild(futureSnapshot, check.path, injector), runCanActivate(futureSnapshot, check.route, injector));
    }), first(result => {
        return result !== true;
    }, true));
}
/**
 * This should fire off `ActivationStart` events for each route being activated at this
 * level.
 * In other words, if you're activating `a` and `b` below, `path` will contain the
 * `ActivatedRouteSnapshot`s for both and we will fire `ActivationStart` for both. Always
 * return
 * `true` so checks continue to run.
 */
function fireActivationStart(snapshot, forwardEvent) {
    if (snapshot !== null && forwardEvent) {
        forwardEvent(new ActivationStart(snapshot));
    }
    return of(true);
}
/**
 * This should fire off `ChildActivationStart` events for each route being activated at this
 * level.
 * In other words, if you're activating `a` and `b` below, `path` will contain the
 * `ActivatedRouteSnapshot`s for both and we will fire `ChildActivationStart` for both. Always
 * return
 * `true` so checks continue to run.
 */
function fireChildActivationStart(snapshot, forwardEvent) {
    if (snapshot !== null && forwardEvent) {
        forwardEvent(new ChildActivationStart(snapshot));
    }
    return of(true);
}
function runCanActivate(futureRSS, futureARS, injector) {
    const canActivate = futureARS.routeConfig ? futureARS.routeConfig.canActivate : null;
    if (!canActivate || canActivate.length === 0)
        return of(true);
    const canActivateObservables = canActivate.map((canActivate) => {
        return defer(() => {
            const closestInjector = getClosestRouteInjector(futureARS) ?? injector;
            const guard = getTokenOrFunctionIdentity(canActivate, closestInjector);
            const guardVal = isCanActivate(guard) ?
                guard.canActivate(futureARS, futureRSS) :
                closestInjector.runInContext(() => guard(futureARS, futureRSS));
            return wrapIntoObservable(guardVal).pipe(first());
        });
    });
    return of(canActivateObservables).pipe(prioritizedGuardValue());
}
function runCanActivateChild(futureRSS, path, injector) {
    const futureARS = path[path.length - 1];
    const canActivateChildGuards = path.slice(0, path.length - 1)
        .reverse()
        .map(p => getCanActivateChild(p))
        .filter(_ => _ !== null);
    const canActivateChildGuardsMapped = canActivateChildGuards.map((d) => {
        return defer(() => {
            const guardsMapped = d.guards.map((canActivateChild) => {
                const closestInjector = getClosestRouteInjector(d.node) ?? injector;
                const guard = getTokenOrFunctionIdentity(canActivateChild, closestInjector);
                const guardVal = isCanActivateChild(guard) ?
                    guard.canActivateChild(futureARS, futureRSS) :
                    closestInjector.runInContext(() => guard(futureARS, futureRSS));
                return wrapIntoObservable(guardVal).pipe(first());
            });
            return of(guardsMapped).pipe(prioritizedGuardValue());
        });
    });
    return of(canActivateChildGuardsMapped).pipe(prioritizedGuardValue());
}
function runCanDeactivate(component, currARS, currRSS, futureRSS, injector) {
    const canDeactivate = currARS && currARS.routeConfig ? currARS.routeConfig.canDeactivate : null;
    if (!canDeactivate || canDeactivate.length === 0)
        return of(true);
    const canDeactivateObservables = canDeactivate.map((c) => {
        const closestInjector = getClosestRouteInjector(currARS) ?? injector;
        const guard = getTokenOrFunctionIdentity(c, closestInjector);
        const guardVal = isCanDeactivate(guard) ?
            guard.canDeactivate(component, currARS, currRSS, futureRSS) :
            closestInjector.runInContext(() => guard(component, currARS, currRSS, futureRSS));
        return wrapIntoObservable(guardVal).pipe(first());
    });
    return of(canDeactivateObservables).pipe(prioritizedGuardValue());
}
export function runCanLoadGuards(injector, route, segments, urlSerializer) {
    const canLoad = route.canLoad;
    if (canLoad === undefined || canLoad.length === 0) {
        return of(true);
    }
    const canLoadObservables = canLoad.map((injectionToken) => {
        const guard = getTokenOrFunctionIdentity(injectionToken, injector);
        const guardVal = isCanLoad(guard) ?
            guard.canLoad(route, segments) :
            injector.runInContext(() => guard(route, segments));
        return wrapIntoObservable(guardVal);
    });
    return of(canLoadObservables)
        .pipe(prioritizedGuardValue(), redirectIfUrlTree(urlSerializer));
}
function redirectIfUrlTree(urlSerializer) {
    return pipe(tap((result) => {
        if (!isUrlTree(result))
            return;
        throw redirectingNavigationError(urlSerializer, result);
    }), map(result => result === true));
}
export function runCanMatchGuards(injector, route, segments, urlSerializer) {
    const canMatch = route.canMatch;
    if (!canMatch || canMatch.length === 0)
        return of(true);
    const canMatchObservables = canMatch.map(injectionToken => {
        const guard = getTokenOrFunctionIdentity(injectionToken, injector);
        const guardVal = isCanMatch(guard) ?
            guard.canMatch(route, segments) :
            injector.runInContext(() => guard(route, segments));
        return wrapIntoObservable(guardVal);
    });
    return of(canMatchObservables)
        .pipe(prioritizedGuardValue(), redirectIfUrlTree(urlSerializer));
}
//# sourceMappingURL=data:application/json;base64,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